<?php

namespace App\Http\Controllers;

use App\Models\Lyric;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class LyricsController extends Controller
{
    public function index()
    {
        // Get flashed data from session if it exists (from generate redirect)
        $lyrics = session('lyrics');
        $input = session('input');
        
        return view('lyrics.index', compact('lyrics', 'input'));
    }

    public function history()
    {
        if (!auth()->check()) {
            return redirect()->route('login')->with('error', 'Please sign in to view your history.');
        }

        $history = auth()->user()->lyrics()->latest()->paginate(12);
        return view('lyrics.history', compact('history'));
    }

    public function generate(Request $request)
    {
        if (!auth()->check()) {
            return redirect()->route('login')->with('error', 'Please sign in to generate lyrics.');
        }

        $request->validate([
            'style' => 'required|string',
            'theme' => 'nullable|string',
            'mood' => 'nullable|string',
            'length' => 'nullable|string',
            'language' => 'nullable|string',
        ]);

        $style = $request->input('style');
        $theme = $request->input('theme', '');
        $mood = $request->input('mood', 'Energetic');
        $length = $request->input('length', 'Medium'); 
        $language = $request->input('language', 'English');

        // Construct the prompt
        $prompt = "Generate song lyrics for a $style song. \n";
        if ($theme) $prompt .= "Theme: $theme\n";
        if ($mood) $prompt .= "Mood: $mood\n";
        if ($length) $prompt .= "Length: $length (include sections like [Verse], [Chorus], etc.)\n";
        if ($language) $prompt .= "Language: $language\n";
        
        $prompt .= "The format must be compatible with Suno AI (use tags like [Intro], [Verse], [Chorus], [Bridge], [Outro]).\n";
        $prompt .= "Do not include any conversational filler, just the lyrics.";

        $apiKey = env('GROQ_API_KEY'); 

        if (!$apiKey) {
            return back()->withErrors(['api_error' => 'Groq API Key is missing in .env file.']);
        }

        try {
            $response = Http::withoutVerifying()->withHeaders([
                'Authorization' => 'Bearer ' . $apiKey,
                'Content-Type' => 'application/json',
            ])->post('https://api.groq.com/openai/v1/chat/completions', [
                'model' => 'llama-3.1-8b-instant',
                'messages' => [
                    ['role' => 'system', 'content' => 'You are a professional songwriter assistant.'],
                    ['role' => 'user', 'content' => $prompt],
                ],
                'temperature' => 0.7,
            ]);

            if ($response->successful()) {
                $data = $response->json();
                $lyrics = $data['choices'][0]['message']['content'] ?? 'No lyrics generated.';
                
                // Save to database associated with user
                auth()->user()->lyrics()->create([
                    'style' => $style,
                    'theme' => $theme,
                    'mood' => $mood,
                    'length' => $length,
                    'language' => $language,
                    'content' => $lyrics,
                ]);

                // Redirect back with the results in the session
                return redirect()->route('home')->with([
                    'lyrics' => $lyrics,
                    'input' => $request->all()
                ]);
            } else {
                Log::error('Groq API Error: ' . $response->body());
                return back()->withErrors(['api_error' => 'Failed to generate lyrics (API error).']);
            }
        } catch (\Exception $e) {
            Log::error('Groq API Exception: ' . $e->getMessage());
            return back()->withErrors(['api_error' => 'An error occurred while connecting to the API.']);
        }
    }
}
