<?php

namespace App\Http\Controllers;

use App\Models\Lyric;
use App\Models\LyricReaction;
use Illuminate\Http\Request;

class CommunityController extends Controller
{
    /**
     * Show all publicly shared lyrics.
     */
    public function index()
    {
        $lyrics = Lyric::with(['user', 'reactions'])
            ->where('is_shared', true)
            ->latest()
            ->paginate(12);

        return view('community.index', compact('lyrics'));
    }

    /**
     * Share or unshare a lyric.
     */
    public function share(Request $request, Lyric $lyric)
    {
        // Only the owner can share their lyric
        if ($lyric->user_id !== auth()->id()) {
            abort(403);
        }

        $lyric->update(['is_shared' => !$lyric->is_shared]);

        $message = $lyric->is_shared
            ? 'Lyric shared to the community!'
            : 'Lyric removed from the community.';

        return back()->with('success', $message);
    }

    /**
     * Like or dislike a lyric. Toggle if same reaction, switch if different.
     */
    public function react(Request $request, Lyric $lyric)
    {
        $request->validate(['type' => 'required|in:like,dislike']);

        $existing = LyricReaction::where('lyric_id', $lyric->id)
            ->where('user_id', auth()->id())
            ->first();

        if ($existing) {
            if ($existing->type === $request->type) {
                // Same reaction — toggle off
                $existing->delete();
                $reacted = null;
            } else {
                // Different reaction — switch it
                $existing->update(['type' => $request->type]);
                $reacted = $request->type;
            }
        } else {
            LyricReaction::create([
                'lyric_id' => $lyric->id,
                'user_id' => auth()->id(),
                'type' => $request->type,
            ]);
            $reacted = $request->type;
        }

        return response()->json([
            'likes' => $lyric->likes()->count(),
            'dislikes' => $lyric->dislikes()->count(),
            'reacted' => $reacted,
        ]);
    }
}
